function [d_stks,psf] = deconv_stk(varargin)
%little function that applies a 2d deconvolution to a stack of images.
%Synatax:   [d_stk,psf] = deconv_stk('edgetaper',1,'deconv','reg','numit',10,...
%               'initpsf',10); 
%Input:     'edgetaper' = taper the image edges according to the PSF.
%               Default = 0; off.
%           'deconv' = type of deconvolution used.  Default = lucy
%               Lucy-Richardson.  Other types: 'reg' = regularized,
%               'wiener' = wiener, and 'blind' = blind, based on maximum
%               likelihood algorithm.
%           'numit' = number of iteration for deconvolution, default = 10.
%               only applicable to blind and Lucy-Richardson.
%           'initpsf' = the size of the initial psf generated for the blind
%               deconvolution.  The PSF restoration is affected strongly by
%               the size of the initial guess INITPSF and less by the 
%               values it contains.
%Output:    d_stk = the deconvoluted image stack
%           psf = the PSF used.

[stks,psf,edgetapered,deconv_type,numit,stks_ch_names] = parse(varargin);

h2 = waitbar(0,'Image Stack: ','position',[20 300 275 50]);    %initialize progress bar.
for i = 1:size(stks,2)      %process each image stack
    waitbar(i/size(stks,2),h2,['Image Stack: ',stks_ch_names{i}]);   %update progress
    if edgetapered        %if the edges need to be tapered
        stk_tmp = edgetaper(stks{1,i},psf);
    else
        stk_tmp = stks{1,i};
    end
    %initiate temporary stack holder
    stk_out = ones(size(stk_tmp));
    %now apply the deconvolution
    switch deconv_type
        case 'lucy'
            h = waitbar(0,'Deconvolving Slice: 1');    %initialize progress bar.
            for j = 1:size(stk_tmp,3)
                stk_out(:,:,j) = deconvlucy(stk_tmp(:,:,j),psf,numit);
                waitbar(j/size(stk_tmp,3),h,['Deconvolving Slice: ',num2str(j+1)]);   %update progress
            end
            close(h)
        case 'reg'
            h = waitbar(0,'Deconvolving Slice: 1');    %initialize progress bar.
            for j = 1:size(stk_tmp,3)
                stk_out(:,:,j) = deconvreg(stk_tmp(:,:,j),psf);
                waitbar(j/size(stk_tmp,3),h,['Deconvolving Slice: ',num2str(j+1)]);   %update progress
            end
            close(h)
        case 'wiener'
            h = waitbar(0,'Deconvolving Slice: 1');    %initialize progress bar.
            for j = 1:size(stk_tmp,3)
                stk_out(:,:,j) = deconvwnr(stk_tmp(:,:,j),psf);
                waitbar(j/size(stk_tmp,3),h,['Deconvolving Slice: ',num2str(j+1)]);   %update progress
            end
            close(h)
        case 'blind'
            initpsf = psf;      %psf is the output so it will change, temp the PSF.
            h = waitbar(0,'Deconvolving Slice: 1');    %initialize progress bar.
            for j = 1:size(stk_tmp,3)
                [stk_out(:,:,j),psf(:,:,j)] = deconvblind(stk_tmp(:,:,j),initpsf,numit);
                waitbar(j/size(stk_tmp,3),h,['Deconvolving Slice: ',num2str(j+1)]);   %update progress
            end
            close(h)
    end
    d_stks{1,i} = stk_out;      %output deconvoluted stack
end
close(h2)

%--------------------------------------------------------------------------
%subfunction to parse the inputs.
function [stks,psf,edgetapered,deconv_type,numit,stks_ch_names] = parse(input)

deconv_type = 'lucy';    %1=lucy-richardson(default), 2=regularized, 3=wiener, 4=blind.
edgetapered = 0;  %off.
numit = 10;     %default # of iteration = 10.
initpsf = 10;  %default = 10x10

%open the image stack first.  Prompt first
prompt_box('title','Open Image Stack','prompt1','Select image stack or stacks ',...
    'prompt2','for deconvolution','position','center');
[stks,img_range,stks_ch_names,channel_paths] = stack_gui(1);  %use cell output mode.

%Parse the input
if ~isempty(input)
    for i = 1:2:size(input,2)
        if ischar(input{1,i});
            switch input{1,i}
                case 'edgetaper'
                    edgetapered = input{1,i+1};
                case 'deconv'
                    if ischar(input{1,i+1})
                        deconv_type = input{1,i+1};
                    else
                        warning(['Your entered deconvolution type is not recognized, reverting to defalut type.']);
                    end
                case 'numit'
                    numit = input{1,i+1};
                case 'initpsf'
                    initpsf = input{1,i+1};
                otherwise
                    warning(['Your input ',input{1,i},' is not recognized.']);
            end
        else
            error(['The parameters you entered is incorrect.  Please check help.']);
        end
    end
end

%open the PSF image.  Prompt first
if ~strcmp(deconv_type,'blind')          %if the deconvolution is not blind open PSF
prompt_box('title','Import the PSF','prompt1','Select the desired PSF ',...
    'prompt2','PSF should be in image form','position','center');
[psf,img_range,channel_names,channel_paths] = stack_gui(1);  %use cell output mode.
psf = im2double(psf{1,1});       %the PSF needs to be in double percision.
else                                %if the deconvolution is blind generate PSF
    psf = fspecial('gaussian',initpsf,10);
end